<?php
include_once get_template_directory() . '/theme-includes.php';

if ( ! function_exists( 'boostup_mikado_styles' ) ) {
	/**
	 * Function that includes theme's core styles
	 */
	function boostup_mikado_styles() {

        $modules_css_deps_array = apply_filters( 'boostup_mikado_filter_modules_css_deps', array() );
		
		//include theme's core styles
		wp_enqueue_style( 'boostup-mikado-default-style', MIKADO_ROOT . '/style.css' );
		wp_enqueue_style( 'boostup-mikado-modules', MIKADO_ASSETS_ROOT . '/css/modules.min.css', $modules_css_deps_array );
		
		boostup_mikado_icon_collections()->enqueueStyles();
		
		wp_enqueue_style( 'wp-mediaelement' );
		
		do_action( 'boostup_mikado_action_enqueue_third_party_styles' );
		
		//is woocommerce installed?
		if ( boostup_mikado_is_woocommerce_installed() && boostup_mikado_load_woo_assets() ) {
			//include theme's woocommerce styles
			wp_enqueue_style( 'boostup-mikado-woo', MIKADO_ASSETS_ROOT . '/css/woocommerce.min.css' );
		}
		
		if ( boostup_mikado_dashboard_page() || boostup_mikado_has_dashboard_shortcodes() ) {
			wp_enqueue_style( 'boostup-mikado-dashboard', MIKADO_FRAMEWORK_ADMIN_ASSETS_ROOT . '/css/mkdf-dashboard.css' );
		}

		$style_dynamic_default_deps = array();

		//define files after which style dynamic needs to be included. It should be included last so it can override other files
        $style_dynamic_deps_array = apply_filters( 'boostup_mikado_filter_style_dynamic_deps', $style_dynamic_default_deps );

        if ( file_exists( MIKADO_ROOT_DIR . '/assets/css/style_dynamic.css' ) && boostup_mikado_is_css_folder_writable() && ! is_multisite() ) {
			wp_enqueue_style( 'boostup-mikado-style-dynamic', MIKADO_ASSETS_ROOT . '/css/style_dynamic.css', $style_dynamic_deps_array, filemtime( MIKADO_ROOT_DIR . '/assets/css/style_dynamic.css' ) ); //it must be included after woocommerce styles so it can override it
		} else if ( file_exists( MIKADO_ROOT_DIR . '/assets/css/style_dynamic_ms_id_' . boostup_mikado_get_multisite_blog_id() . '.css' ) && boostup_mikado_is_css_folder_writable() && is_multisite() ) {
			wp_enqueue_style( 'boostup-mikado-style-dynamic', MIKADO_ASSETS_ROOT . '/css/style_dynamic_ms_id_' . boostup_mikado_get_multisite_blog_id() . '.css', $style_dynamic_deps_array, filemtime( MIKADO_ROOT_DIR . '/assets/css/style_dynamic_ms_id_' . boostup_mikado_get_multisite_blog_id() . '.css' ) ); //it must be included after woocommerce styles so it can override it
		}
		
		//is responsive option turned on?
		if ( boostup_mikado_is_responsive_on() ) {
			wp_enqueue_style( 'boostup-mikado-modules-responsive', MIKADO_ASSETS_ROOT . '/css/modules-responsive.min.css' );
			
			//is woocommerce installed?
			if ( boostup_mikado_is_woocommerce_installed() && boostup_mikado_load_woo_assets() ) {
				//include theme's woocommerce responsive styles
				wp_enqueue_style( 'boostup-mikado-woo-responsive', MIKADO_ASSETS_ROOT . '/css/woocommerce-responsive.min.css' );
			}
			
			//include proper styles
			if ( file_exists( MIKADO_ROOT_DIR . '/assets/css/style_dynamic_responsive.css' ) && boostup_mikado_is_css_folder_writable() && ! is_multisite() ) {
				wp_enqueue_style( 'boostup-mikado-style-dynamic-responsive', MIKADO_ASSETS_ROOT . '/css/style_dynamic_responsive.css', array(), filemtime( MIKADO_ROOT_DIR . '/assets/css/style_dynamic_responsive.css' ) );
			} else if ( file_exists( MIKADO_ROOT_DIR . '/assets/css/style_dynamic_responsive_ms_id_' . boostup_mikado_get_multisite_blog_id() . '.css' ) && boostup_mikado_is_css_folder_writable() && is_multisite() ) {
				wp_enqueue_style( 'boostup-mikado-style-dynamic-responsive', MIKADO_ASSETS_ROOT . '/css/style_dynamic_responsive_ms_id_' . boostup_mikado_get_multisite_blog_id() . '.css', array(), filemtime( MIKADO_ROOT_DIR . '/assets/css/style_dynamic_responsive_ms_id_' . boostup_mikado_get_multisite_blog_id() . '.css' ) );
			}
		}
	}
	
	add_action( 'wp_enqueue_scripts', 'boostup_mikado_styles' );
}

if ( ! function_exists( 'boostup_mikado_google_fonts_styles' ) ) {
	/**
	 * Function that includes google fonts defined anywhere in the theme
	 */
	function boostup_mikado_google_fonts_styles() {
		$font_simple_field_array = boostup_mikado_options()->getOptionsByType( 'fontsimple' );
		if ( ! ( is_array( $font_simple_field_array ) && count( $font_simple_field_array ) > 0 ) ) {
			$font_simple_field_array = array();
		}
		
		$font_field_array = boostup_mikado_options()->getOptionsByType( 'font' );
		if ( ! ( is_array( $font_field_array ) && count( $font_field_array ) > 0 ) ) {
			$font_field_array = array();
		}
		
		$available_font_options = array_merge( $font_simple_field_array, $font_field_array );
		
		$google_font_weight_array = boostup_mikado_options()->getOptionValue( 'google_font_weight' );
		if ( ! empty( $google_font_weight_array ) ) {
			$google_font_weight_array = array_slice( boostup_mikado_options()->getOptionValue( 'google_font_weight' ), 1 );
		}
		
		$font_weight_str = '300,400,500,600,700';
		if ( ! empty( $google_font_weight_array ) && $google_font_weight_array !== '' ) {
			$font_weight_str = implode( ',', $google_font_weight_array );
		}
		
		$google_font_subset_array = boostup_mikado_options()->getOptionValue( 'google_font_subset' );
		if ( ! empty( $google_font_subset_array ) ) {
			$google_font_subset_array = array_slice( boostup_mikado_options()->getOptionValue( 'google_font_subset' ), 1 );
		}
		
		$font_subset_str = 'latin-ext';
		if ( ! empty( $google_font_subset_array ) && $google_font_subset_array !== '' ) {
			$font_subset_str = implode( ',', $google_font_subset_array );
		}
		
		//default fonts
		$default_font_family = array(
			'Josefin Sans',
            'Montserrat'
		);
		
		$modified_default_font_family = array();
		foreach ( $default_font_family as $default_font ) {
			$modified_default_font_family[] = $default_font . ':' . str_replace( ' ', '', $font_weight_str );
		}
		
		$default_font_string = implode( '|', $modified_default_font_family );
		
		//define available font options array
		$fonts_array = array();
		foreach ( $available_font_options as $font_option ) {
			//is font set and not set to default and not empty?
			$font_option_value = boostup_mikado_options()->getOptionValue( $font_option );
			
			if ( boostup_mikado_is_font_option_valid( $font_option_value ) && ! boostup_mikado_is_native_font( $font_option_value ) ) {
				$font_option_string = $font_option_value . ':' . $font_weight_str;
				
				if ( ! in_array( str_replace( '+', ' ', $font_option_value ), $default_font_family ) && ! in_array( $font_option_string, $fonts_array ) ) {
					$fonts_array[] = $font_option_string;
				}
			}
		}
		
		$fonts_array         = array_diff( $fonts_array, array( '-1:' . $font_weight_str ) );
		$google_fonts_string = implode( '|', $fonts_array );
		
		$protocol = is_ssl() ? 'https:' : 'http:';
		
		//is google font option checked anywhere in theme?
		if ( count( $fonts_array ) > 0 ) {
			
			//include all checked fonts
			$fonts_full_list      = $default_font_string . '|' . str_replace( '+', ' ', $google_fonts_string );
			$fonts_full_list_args = array(
				'family' => urlencode( $fonts_full_list ),
				'subset' => urlencode( $font_subset_str ),
			);
			
			$boostup_mikado_global_fonts = add_query_arg( $fonts_full_list_args, $protocol . '//fonts.googleapis.com/css' );
			wp_enqueue_style( 'boostup-mikado-google-fonts', esc_url_raw( $boostup_mikado_global_fonts ), array(), '1.0.0' );
			
		} else {
			//include default google font that theme is using
			$default_fonts_args          = array(
				'family' => urlencode( $default_font_string ),
				'subset' => urlencode( $font_subset_str ),
			);
			$boostup_mikado_global_fonts = add_query_arg( $default_fonts_args, $protocol . '//fonts.googleapis.com/css' );
			wp_enqueue_style( 'boostup-mikado-google-fonts', esc_url_raw( $boostup_mikado_global_fonts ), array(), '1.0.0' );
		}
	}
	
	add_action( 'wp_enqueue_scripts', 'boostup_mikado_google_fonts_styles' );
}

if ( ! function_exists( 'boostup_mikado_scripts' ) ) {
	/**
	 * Function that includes all necessary scripts
	 */
	function boostup_mikado_scripts() {
		global $wp_scripts;
		
		//init theme core scripts
		wp_enqueue_script( 'jquery-ui-core' );
		wp_enqueue_script( 'jquery-ui-tabs' );
		wp_enqueue_script( 'jquery-ui-accordion' );
		wp_enqueue_script( 'wp-mediaelement' );
		
		// 3rd party JavaScripts that we used in our theme
		wp_enqueue_script( 'appear', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.appear.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'modernizr', MIKADO_ASSETS_ROOT . '/js/modules/plugins/modernizr.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'hoverIntent', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.hoverIntent.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'jquery-plugin', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.plugin.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'owl-carousel', MIKADO_ASSETS_ROOT . '/js/modules/plugins/owl.carousel.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'waypoints', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.waypoints.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'fluidvids', MIKADO_ASSETS_ROOT . '/js/modules/plugins/fluidvids.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'perfect-scrollbar', MIKADO_ASSETS_ROOT . '/js/modules/plugins/perfect-scrollbar.jquery.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'ScrollToPlugin', MIKADO_ASSETS_ROOT . '/js/modules/plugins/ScrollToPlugin.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'parallax', MIKADO_ASSETS_ROOT . '/js/modules/plugins/parallax.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'waitforimages', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.waitforimages.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'prettyphoto', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.prettyPhoto.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'jquery-easing-1.3', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.easing.1.3.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'isotope', MIKADO_ASSETS_ROOT . '/js/modules/plugins/isotope.pkgd.min.js', array( 'jquery' ), false, true );
		wp_enqueue_script( 'packery', MIKADO_ASSETS_ROOT . '/js/modules/plugins/packery-mode.pkgd.min.js', array( 'jquery' ), false, true );
		
		do_action( 'boostup_mikado_action_enqueue_third_party_scripts' );

		if ( boostup_mikado_is_woocommerce_installed() ) {
			wp_enqueue_script( 'select2' );
		}

		if ( boostup_mikado_is_page_smooth_scroll_enabled() ) {
			wp_enqueue_script( 'tweenLite', MIKADO_ASSETS_ROOT . '/js/modules/plugins/TweenLite.min.js', array( 'jquery' ), false, true );
			wp_enqueue_script( 'smoothPageScroll', MIKADO_ASSETS_ROOT . '/js/modules/plugins/smoothPageScroll.js', array( 'jquery' ), false, true );
		}

		//include google map api script
		$google_maps_api_key          = boostup_mikado_options()->getOptionValue( 'google_maps_api_key' );
		$google_maps_extensions       = '';
		$google_maps_extensions_array = apply_filters( 'boostup_mikado_filter_google_maps_extensions_array', array() );

		if ( ! empty( $google_maps_extensions_array ) ) {
			$google_maps_extensions .= '&libraries=';
			$google_maps_extensions .= implode( ',', $google_maps_extensions_array );
		}

		if ( ! empty( $google_maps_api_key ) ) {
			wp_enqueue_script( 'boostup-mikado-google-map-api', '//maps.googleapis.com/maps/api/js?key=' . esc_attr( $google_maps_api_key ) . $google_maps_extensions, array(), false, true );
            if ( ! empty( $google_maps_extensions_array ) && is_array( $google_maps_extensions_array ) ) {
                wp_enqueue_script('geocomplete', MIKADO_ASSETS_ROOT . '/js/modules/plugins/jquery.geocomplete.min.js', array('jquery', 'boostup-mikado-google-map-api'), false, true);
            }
		}

		wp_enqueue_script( 'boostup-mikado-modules', MIKADO_ASSETS_ROOT . '/js/modules.min.js', array( 'jquery' ), false, true );
		
		if ( boostup_mikado_dashboard_page() || boostup_mikado_has_dashboard_shortcodes() ) {
			$dash_array_deps = array(
				'jquery-ui-datepicker',
				'jquery-ui-sortable'
			);
			
			wp_enqueue_script( 'boostup-mikado-dashboard', MIKADO_FRAMEWORK_ADMIN_ASSETS_ROOT . '/js/mkdf-dashboard.js', $dash_array_deps, false, true );
			
			wp_enqueue_script( 'wp-util' );
			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_script( 'iris', admin_url( 'js/iris.min.js' ), array( 'jquery-ui-draggable', 'jquery-ui-slider', 'jquery-touch-punch' ), false, 1 );
			wp_enqueue_script( 'wp-color-picker', admin_url( 'js/color-picker.min.js' ), array( 'iris' ), false, 1 );
			
			$colorpicker_l10n = array(
				'clear'         => esc_html__( 'Clear', 'boostup' ),
				'defaultString' => esc_html__( 'Default', 'boostup' ),
				'pick'          => esc_html__( 'Select Color', 'boostup' ),
				'current'       => esc_html__( 'Current Color', 'boostup' ),
			);
			
			wp_localize_script( 'wp-color-picker', 'wpColorPickerL10n', $colorpicker_l10n );
		}

		//include comment reply script
		$wp_scripts->add_data( 'comment-reply', 'group', 1 );
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}

	add_action( 'wp_enqueue_scripts', 'boostup_mikado_scripts' );
}

if ( ! function_exists( 'boostup_mikado_theme_setup' ) ) {
	/**
	 * Function that adds various features to theme. Also defines image sizes that are used in a theme
	 */
	function boostup_mikado_theme_setup() {
		//add support for feed links
		add_theme_support( 'automatic-feed-links' );

		//add support for post formats
		add_theme_support( 'post-formats', array( 'gallery', 'link', 'quote', 'video', 'audio' ) );

		//add theme support for post thumbnails
		add_theme_support( 'post-thumbnails' );

		//add theme support for title tag
		add_theme_support( 'title-tag' );

        //add theme support for editor style
        add_editor_style( 'framework/admin/assets/css/editor-style.css' );

		//defined content width variable
		$GLOBALS['content_width'] = apply_filters( 'boostup_mikado_filter_set_content_width', 1100 );

		//define thumbnail sizes
		add_image_size( 'boostup_mikado_image_square', 650, 650, true );
		add_image_size( 'boostup_mikado_image_landscape', 1300, 650, true );
		add_image_size( 'boostup_mikado_image_portrait', 650, 1300, true );
		add_image_size( 'boostup_mikado_image_huge', 1300, 1300, true );

		load_theme_textdomain( 'boostup', get_template_directory() . '/languages' );
	}

	add_action( 'after_setup_theme', 'boostup_mikado_theme_setup' );
}

if ( ! function_exists( 'boostup_mikado_enqueue_editor_customizer_styles' ) ) {
	/**
	 * Enqueue supplemental block editor styles
	 */
	function boostup_mikado_enqueue_editor_customizer_styles() {
		wp_enqueue_style( 'themename-style-modules-admin-styles', MIKADO_FRAMEWORK_ADMIN_ASSETS_ROOT . '/css/mkdf-modules-admin.css' );
		wp_enqueue_style( 'themename-style-handle-editor-customizer-styles', MIKADO_FRAMEWORK_ADMIN_ASSETS_ROOT . '/css/editor-customizer-style.css' );
	}

	// add google font
	add_action( 'enqueue_block_editor_assets', 'boostup_mikado_google_fonts_styles' );
	// add action
	add_action( 'enqueue_block_editor_assets', 'boostup_mikado_enqueue_editor_customizer_styles' );
}

if ( ! function_exists( 'boostup_mikado_is_responsive_on' ) ) {
	/**
	 * Checks whether responsive mode is enabled in theme options
	 * @return bool
	 */
	function boostup_mikado_is_responsive_on() {
		return boostup_mikado_options()->getOptionValue( 'responsiveness' ) !== 'no';
	}
}

if ( ! function_exists( 'boostup_mikado_rgba_color' ) ) {
	/**
	 * Function that generates rgba part of css color property
	 *
	 * @param $color string hex color
	 * @param $transparency float transparency value between 0 and 1
	 *
	 * @return string generated rgba string
	 */
	function boostup_mikado_rgba_color( $color, $transparency ) {
		if ( $color !== '' && $transparency !== '' ) {
			$rgba_color = '';

			$rgb_color_array = boostup_mikado_hex2rgb( $color );
			$rgba_color      .= 'rgba(' . implode( ', ', $rgb_color_array ) . ', ' . $transparency . ')';

			return $rgba_color;
		}
	}
}

if ( ! function_exists( 'boostup_mikado_header_meta' ) ) {
	/**
	 * Function that echoes meta data if our seo is enabled
	 */
	function boostup_mikado_header_meta() { ?>

		<meta charset="<?php bloginfo( 'charset' ); ?>"/>
		<link rel="profile" href="http://gmpg.org/xfn/11"/>
		<?php if ( is_singular() && pings_open( get_queried_object() ) ) : ?>
			<link rel="pingback" href="<?php bloginfo( 'pingback_url' ); ?>">
		<?php endif; ?>

	<?php }

	add_action( 'boostup_mikado_action_header_meta', 'boostup_mikado_header_meta' );
}

if ( ! function_exists( 'boostup_mikado_user_scalable_meta' ) ) {
	/**
	 * Function that outputs user scalable meta if responsiveness is turned on
	 * Hooked to boostup_mikado_action_header_meta action
	 */
	function boostup_mikado_user_scalable_meta() {
		//is responsiveness option is chosen?
		if ( boostup_mikado_is_responsive_on() ) { ?>
			<meta name="viewport" content="width=device-width,initial-scale=1,user-scalable=yes">
		<?php } else { ?>
			<meta name="viewport" content="width=1200,user-scalable=yes">
		<?php }
	}

	add_action( 'boostup_mikado_action_header_meta', 'boostup_mikado_user_scalable_meta' );
}

if ( ! function_exists( 'boostup_mikado_smooth_page_transitions' ) ) {
	/**
	 * Function that outputs smooth page transitions html if smooth page transitions functionality is turned on
	 * Hooked to boostup_mikado_action_after_body_tag action
	 */
	function boostup_mikado_smooth_page_transitions() {
		$id = boostup_mikado_get_page_id();

		if ( boostup_mikado_get_meta_field_intersect( 'smooth_page_transitions', $id ) === 'yes' && boostup_mikado_get_meta_field_intersect( 'page_transition_preloader', $id ) === 'yes' ) { ?>
			<div class="mkdf-smooth-transition-loader mkdf-mimic-ajax">
				<div class="mkdf-st-loader">
					<div class="mkdf-st-loader1">
						<?php boostup_mikado_loading_spinners(); ?>
					</div>
				</div>
			</div>
		<?php }
	}

	add_action( 'boostup_mikado_action_after_body_tag', 'boostup_mikado_smooth_page_transitions', 10 );
}

if ( ! function_exists( 'boostup_mikado_back_to_top_button' ) ) {
	/**
	 * Function that outputs back to top button html if back to top functionality is turned on
	 * Hooked to boostup_mikado_action_after_wrapper_inner action
	 */
	function boostup_mikado_back_to_top_button() {
		if ( boostup_mikado_options()->getOptionValue( 'show_back_button' ) == 'yes' ) { ?>
			<a id='mkdf-back-to-top' href='#'>
				<div class="mkdf-icon-stack-outer">

                <?php if ( boostup_mikado_options()->getOptionValue( 'show_back_button_svg' ) == 'yes' ) { ?>

                    <div class="mkdf-icon-stack">
                    	<span class="mkdf-icon-stack-inner">
                    <svg xmlns="http://www.w3.org/2000/svg" width="33.371px"
                         height="55px" viewBox="0 0 33.371 48.833" enable-background="new 0 0 33.371 48.833">

                        <path fill="#ea3d56" d="M32.643,22.556c-0.417,0.543-3.237,1.685-3.237,1.685c-0.679,0.381-1.462,0.707-2.141,1.142
	c-0.157,0.108-0.209,0.272-0.314,0.435c-0.261,0.543-0.626,1.033-0.992,1.522c-0.679,0.869-1.514,1.086-2.507,0.924
	c-0.261,0-0.417-0.272-0.313-0.598c0.052-0.326,0.157-0.598,0.417-0.815c0.157-0.218,0.366-0.434,0.575-0.598
	c0.261-0.217,0.522-0.38,0.731-0.597c0.209-0.109,0.313-0.272,0.418-0.489c0.417-1.033,0.574-2.121,0.47-3.261
	c0-0.055-0.052-0.109-0.052-0.164c-0.104,0-0.209,0.055-0.261,0.109l-0.157,0.163c-0.992,1.304-1.984,2.61-2.872,3.968
	c-0.366,0.652-2.611,6.414-3.133,8.099c-0.626,2.012-1.41,4.022-2.246,5.979c-0.574,1.25-1.096,2.555-1.775,3.751l-0.575,0.978
	c-0.261,0.327-1.515,2.174-2.35,2.773c-1.306,0.924-1.724,0.163-1.828-0.652v-0.544V46.31c0-0.218,0-0.435,0.052-0.652
	c0.366-3.153,1.567-5.98,2.872-8.698c1.044-2.228,2.141-4.402,3.342-6.577c0.105-0.163,1.202-2.174,1.828-3.207l0.543-1.38
	l-0.019,0.022c-0.261,0.435-2.193,2.391-2.246,2.446l-0.888,0.924c-0.261,0.218-0.47,0.435-0.731,0.598
	c-0.627,0.489-1.41,0.706-2.193,0.652c-0.209,0-0.417,0.054-0.626,0c-0.261,0-0.47,0-0.731-0.109
	c-0.417-0.163-0.679-0.435-0.835-0.924c-0.104-0.489-0.157-0.924-0.104-1.468c0.104-1.848,0.522-3.696,1.148-5.436
	c0.052-0.271,0.209-0.543,0.157-0.815c-0.313,0.218-0.522,0.597-0.783,0.924c-0.888,1.25-1.775,2.5-2.768,3.642
	c-0.887,1.088-1.932,2.012-3.028,2.881c-0.992,0.761-2.088,0.978-3.237,0.815c-0.47-0.054-0.836-0.381-0.992-0.87
	C1.104,28.589,1,28.045,1,27.502c0-1.74,0.261-3.425,0.626-5.11c0.418-2.011,1.044-3.967,1.776-5.87
	c1.044-2.773,2.193-5.544,3.446-8.208c0.94-2.011,1.932-4.022,3.133-5.871l1.044-1.413c0.104-0.109,0.261-0.164,0.366-0.272
	l0.104,0.055c-0.052,0.435,0.157,0.163,0.313,0.163l0.783-0.163c0.522-0.109,0.888,0.163,1.253,0.49
	c0.209,0.217,0.366,0.434,0.574,0.706c0.053,0.055,0.104,0.218,0.104,0.327c-0.052,0.271-0.104,0.597-0.261,0.924
	c-0.366,0.87-0.835,1.685-1.253,2.555c-1.044,2.12-2.089,4.348-2.977,6.577c-0.835,1.957-1.619,3.967-2.402,5.979
	c-1.097,2.772-2.141,5.544-3.186,8.371c-0.156,0.381-0.261,0.761-0.365,1.142c-0.052,0.163-0.052,0.381-0.052,0.543
	c0.208,0.055,0.365-0.109,0.47-0.218c0.47-0.326,0.992-0.706,1.462-1.086c1.044-0.924,1.984-1.957,2.82-3.098
	c1.41-1.848,2.768-3.697,4.021-5.653c1.306-1.957,2.455-3.914,3.342-6.143c0.104-0.326,0.313-0.598,0.574-0.815
	c0.104-0.054,0.209-0.054,0.261,0.109c0.052,0.163,0.157,0.271,0.313,0.163c0.366-0.271,0.73-0.271,1.096-0.217
	c0.418,0,0.783,0.271,1.044,0.543l0.731,0.815c0.157,0.163,0.209,0.327,0.209,0.543c-0.105,0.38-0.157,0.761-0.261,1.142
	c-0.313,0.707-0.575,1.413-0.94,2.12c-1.253,2.609-2.297,5.327-3.185,8.153c-0.261,0.87-0.575,1.739-0.888,2.609
	c-0.052,0.163-0.157,0.327-0.157,0.489c-0.052,0.054,0,0.163,0,0.218h0.261c0.836-0.326,1.567-1.033,2.141-1.685
	c0.996-1.141,2.734-3.248,2.985-3.537c0.581-1.371,1.191-2.736,1.87-4.074l0.157-0.38c0.104-0.218,0.261-0.327,0.522-0.327
	c0.366,0,0.627,0.218,0.627,0.599c0.052,0.163,0,0.38,0,0.543c-0.157,0.599-0.261,1.142-0.366,1.739v0.109
	c0.052-0.054,0.104-0.054,0.104-0.054l0.47-0.489c0.261-0.163,0.47-0.38,0.731-0.543c0.313-0.218,0.626-0.327,0.992-0.271
	c0.208,0.054,0.47,0.054,0.679,0.054c0.156,0.055,0.366,0.109,0.522,0.218c0.575,0.38,0.94,0.924,1.097,1.63
	c0.104,0.544,0.104,1.142,0,1.685v0.381c0,0,0.052,0.055,0.104,0.055c0.157-0.055,0.261-0.163,0.418-0.218
	c0.73-0.38,2.715-1.304,3.603-1.413C32.121,22.121,32.8,22.338,32.643,22.556z"/>
                    </svg>
                    </span>
                    </div>


                <?php } else { ?>

                <span class="mkdf-icon-stack">
                	<span class="mkdf-icon-stack-inner">
                     	<?php boostup_mikado_icon_collections()->getBackToTopIcon( 'font_awesome' ); ?>
                 	</span>
                </span>

                <?php } ?>
				</div>
			 <span class="mkdf-back-to-top-inner">
                    <span class="mkdf-back-to-top-icon-back"> 
                    	<?php if ( boostup_mikado_options()->getOptionValue( 'show_back_button_svg' ) == 'yes' ) { ?>

                   
                    <svg xmlns="http://www.w3.org/2000/svg" width="33.371px"
                         height="55px" viewBox="0 0 33.371 48.833" enable-background="new 0 0 33.371 48.833">

                        <path fill="#ea3d56" d="M32.643,22.556c-0.417,0.543-3.237,1.685-3.237,1.685c-0.679,0.381-1.462,0.707-2.141,1.142
	c-0.157,0.108-0.209,0.272-0.314,0.435c-0.261,0.543-0.626,1.033-0.992,1.522c-0.679,0.869-1.514,1.086-2.507,0.924
	c-0.261,0-0.417-0.272-0.313-0.598c0.052-0.326,0.157-0.598,0.417-0.815c0.157-0.218,0.366-0.434,0.575-0.598
	c0.261-0.217,0.522-0.38,0.731-0.597c0.209-0.109,0.313-0.272,0.418-0.489c0.417-1.033,0.574-2.121,0.47-3.261
	c0-0.055-0.052-0.109-0.052-0.164c-0.104,0-0.209,0.055-0.261,0.109l-0.157,0.163c-0.992,1.304-1.984,2.61-2.872,3.968
	c-0.366,0.652-2.611,6.414-3.133,8.099c-0.626,2.012-1.41,4.022-2.246,5.979c-0.574,1.25-1.096,2.555-1.775,3.751l-0.575,0.978
	c-0.261,0.327-1.515,2.174-2.35,2.773c-1.306,0.924-1.724,0.163-1.828-0.652v-0.544V46.31c0-0.218,0-0.435,0.052-0.652
	c0.366-3.153,1.567-5.98,2.872-8.698c1.044-2.228,2.141-4.402,3.342-6.577c0.105-0.163,1.202-2.174,1.828-3.207l0.543-1.38
	l-0.019,0.022c-0.261,0.435-2.193,2.391-2.246,2.446l-0.888,0.924c-0.261,0.218-0.47,0.435-0.731,0.598
	c-0.627,0.489-1.41,0.706-2.193,0.652c-0.209,0-0.417,0.054-0.626,0c-0.261,0-0.47,0-0.731-0.109
	c-0.417-0.163-0.679-0.435-0.835-0.924c-0.104-0.489-0.157-0.924-0.104-1.468c0.104-1.848,0.522-3.696,1.148-5.436
	c0.052-0.271,0.209-0.543,0.157-0.815c-0.313,0.218-0.522,0.597-0.783,0.924c-0.888,1.25-1.775,2.5-2.768,3.642
	c-0.887,1.088-1.932,2.012-3.028,2.881c-0.992,0.761-2.088,0.978-3.237,0.815c-0.47-0.054-0.836-0.381-0.992-0.87
	C1.104,28.589,1,28.045,1,27.502c0-1.74,0.261-3.425,0.626-5.11c0.418-2.011,1.044-3.967,1.776-5.87
	c1.044-2.773,2.193-5.544,3.446-8.208c0.94-2.011,1.932-4.022,3.133-5.871l1.044-1.413c0.104-0.109,0.261-0.164,0.366-0.272
	l0.104,0.055c-0.052,0.435,0.157,0.163,0.313,0.163l0.783-0.163c0.522-0.109,0.888,0.163,1.253,0.49
	c0.209,0.217,0.366,0.434,0.574,0.706c0.053,0.055,0.104,0.218,0.104,0.327c-0.052,0.271-0.104,0.597-0.261,0.924
	c-0.366,0.87-0.835,1.685-1.253,2.555c-1.044,2.12-2.089,4.348-2.977,6.577c-0.835,1.957-1.619,3.967-2.402,5.979
	c-1.097,2.772-2.141,5.544-3.186,8.371c-0.156,0.381-0.261,0.761-0.365,1.142c-0.052,0.163-0.052,0.381-0.052,0.543
	c0.208,0.055,0.365-0.109,0.47-0.218c0.47-0.326,0.992-0.706,1.462-1.086c1.044-0.924,1.984-1.957,2.82-3.098
	c1.41-1.848,2.768-3.697,4.021-5.653c1.306-1.957,2.455-3.914,3.342-6.143c0.104-0.326,0.313-0.598,0.574-0.815
	c0.104-0.054,0.209-0.054,0.261,0.109c0.052,0.163,0.157,0.271,0.313,0.163c0.366-0.271,0.73-0.271,1.096-0.217
	c0.418,0,0.783,0.271,1.044,0.543l0.731,0.815c0.157,0.163,0.209,0.327,0.209,0.543c-0.105,0.38-0.157,0.761-0.261,1.142
	c-0.313,0.707-0.575,1.413-0.94,2.12c-1.253,2.609-2.297,5.327-3.185,8.153c-0.261,0.87-0.575,1.739-0.888,2.609
	c-0.052,0.163-0.157,0.327-0.157,0.489c-0.052,0.054,0,0.163,0,0.218h0.261c0.836-0.326,1.567-1.033,2.141-1.685
	c0.996-1.141,2.734-3.248,2.985-3.537c0.581-1.371,1.191-2.736,1.87-4.074l0.157-0.38c0.104-0.218,0.261-0.327,0.522-0.327
	c0.366,0,0.627,0.218,0.627,0.599c0.052,0.163,0,0.38,0,0.543c-0.157,0.599-0.261,1.142-0.366,1.739v0.109
	c0.052-0.054,0.104-0.054,0.104-0.054l0.47-0.489c0.261-0.163,0.47-0.38,0.731-0.543c0.313-0.218,0.626-0.327,0.992-0.271
	c0.208,0.054,0.47,0.054,0.679,0.054c0.156,0.055,0.366,0.109,0.522,0.218c0.575,0.38,0.94,0.924,1.097,1.63
	c0.104,0.544,0.104,1.142,0,1.685v0.381c0,0,0.052,0.055,0.104,0.055c0.157-0.055,0.261-0.163,0.418-0.218
	c0.73-0.38,2.715-1.304,3.603-1.413C32.121,22.121,32.8,22.338,32.643,22.556z"/>
                    </svg>
                   


                <?php } else { ?>

                <span class="">
                     <?php boostup_mikado_icon_collections()->getBackToTopIcon( 'font_awesome' ); ?>
                </span>

                <?php } ?>

                    </span>
                </span>
			</a>
		<?php }
	}
	
	add_action( 'boostup_mikado_action_after_wrapper_inner', 'boostup_mikado_back_to_top_button', 30 );
}

if ( ! function_exists( 'boostup_mikado_get_page_id' ) ) {
	/**
	 * Function that returns current page / post id.
	 * Checks if current page is woocommerce page and returns that id if it is.
	 * Checks if current page is any archive page (category, tag, date, author etc.) and returns -1 because that isn't
	 * page that is created in WP admin.
	 *
	 * @return int
	 *
	 * @version 0.1
	 *
	 * @see boostup_mikado_is_woocommerce_installed()
	 * @see boostup_mikado_is_woocommerce_shop()
	 */
	function boostup_mikado_get_page_id() {
		if ( boostup_mikado_is_woocommerce_installed() && boostup_mikado_is_woocommerce_shop() ) {
			return boostup_mikado_get_woo_shop_page_id();
		}

		if ( boostup_mikado_is_default_wp_template() ) {
			return - 1;
		}

		return get_queried_object_id();
	}
}

if ( ! function_exists( 'boostup_mikado_get_multisite_blog_id' ) ) {
	/**
	 * Check is multisite and return blog id
	 *
	 * @return int
	 */
	function boostup_mikado_get_multisite_blog_id() {
		if ( is_multisite() ) {
			return get_blog_details()->blog_id;
		}
	}
}

if ( ! function_exists( 'boostup_mikado_is_default_wp_template' ) ) {
	/**
	 * Function that checks if current page archive page, search, 404 or default home blog page
	 * @return bool
	 *
	 * @see is_archive()
	 * @see is_search()
	 * @see is_404()
	 * @see is_front_page()
	 * @see is_home()
	 */
	function boostup_mikado_is_default_wp_template() {
		return is_archive() || is_search() || is_404() || ( is_front_page() && is_home() );
	}
}

if ( ! function_exists( 'boostup_mikado_has_shortcode' ) ) {
	/**
	 * Function that checks whether shortcode exists on current page / post
	 *
	 * @param string shortcode to find
	 * @param string content to check. If isn't passed current post content will be used
	 *
	 * @return bool whether content has shortcode or not
	 */
	function boostup_mikado_has_shortcode( $shortcode, $content = '' ) {
		$has_shortcode = false;

		if ( $shortcode ) {
			//if content variable isn't past
			if ( $content == '' ) {
				//take content from current post
				$page_id = boostup_mikado_get_page_id();
				if ( ! empty( $page_id ) ) {
					$current_post = get_post( $page_id );

					if ( is_object( $current_post ) && property_exists( $current_post, 'post_content' ) ) {
						$content = $current_post->post_content;
					}
				}
			}

			//does content has shortcode added?
			if( has_shortcode( $content, $shortcode ) ) {
				$has_shortcode = true;
			}
		}

		return $has_shortcode;
	}
}

if ( ! function_exists( 'boostup_mikado_get_unique_page_class' ) ) {
	/**
	 * Returns unique page class based on post type and page id
	 *
	 * $params int $id is page id
	 * $params bool $allowSingleProductOption
	 * @return string
	 */
	function boostup_mikado_get_unique_page_class( $id, $allowSingleProductOption = false ) {
		$page_class = '';

		if ( boostup_mikado_is_woocommerce_installed() && $allowSingleProductOption ) {

			if ( is_product() ) {
				$id = get_the_ID();
			}
		}

		if ( is_single() ) {
			$page_class = '.postid-' . $id;
		} elseif ( is_home() ) {
			$page_class .= '.home';
		} elseif ( is_archive() || $id === boostup_mikado_get_woo_shop_page_id() ) {
			$page_class .= '.archive';
		} elseif ( is_search() ) {
			$page_class .= '.search';
		} elseif ( is_404() ) {
			$page_class .= '.error404';
		} else {
			$page_class .= '.page-id-' . $id;
		}

		return $page_class;
	}
}

if ( ! function_exists( 'boostup_mikado_page_custom_style' ) ) {
	/**
	 * Function that print custom page style
	 */
	function boostup_mikado_page_custom_style() {
		$style = apply_filters( 'boostup_mikado_filter_add_page_custom_style', $style = '' );

		if ( $style !== '' ) {

            $page_css_deps_array = array(
                'boostup-mikado-modules',
                'boostup-mikado-style-dynamic'
            );
            if (boostup_mikado_is_woocommerce_installed() && boostup_mikado_load_woo_assets()) {
                $page_css_deps_array[] = 'boostup-mikado-woo';
            }
            wp_enqueue_style('boostup-mikado-page', MIKADO_ASSETS_ROOT . '/css/page.css', $page_css_deps_array);

            wp_add_inline_style( 'boostup-mikado-page', $style );
		}
	}

	add_action( 'wp_enqueue_scripts', 'boostup_mikado_page_custom_style' );
}

if ( ! function_exists( 'boostup_mikado_print_custom_js' ) ) {
	/**
	 * Prints out custom css from theme options
	 */
	function boostup_mikado_print_custom_js() {
		$custom_js = boostup_mikado_options()->getOptionValue( 'custom_js' );

		if ( ! empty( $custom_js ) ) {
			wp_add_inline_script( 'boostup-mikado-modules', $custom_js );
		}
	}

	add_action( 'wp_enqueue_scripts', 'boostup_mikado_print_custom_js' );
}

if ( ! function_exists( 'boostup_mikado_get_global_variables' ) ) {
	/**
	 * Function that generates global variables and put them in array so they could be used in the theme
	 */
	function boostup_mikado_get_global_variables() {
		$global_variables = array();
		
		$global_variables['mkdfAddForAdminBar']      = is_admin_bar_showing() ? 32 : 0;
		$global_variables['mkdfElementAppearAmount'] = -100;
		$global_variables['mkdfAjaxUrl']             = esc_url( admin_url( 'admin-ajax.php' ) );
		$global_variables['sliderNavPrevArrow']       = 'arrow_carrot-left';
		$global_variables['sliderNavNextArrow']       = 'arrow_carrot-right';
		
		$global_variables = apply_filters( 'boostup_mikado_filter_js_global_variables', $global_variables );
		
		wp_localize_script( 'boostup-mikado-modules', 'mkdfGlobalVars', array(
			'vars' => $global_variables
		) );
	}

	add_action( 'wp_enqueue_scripts', 'boostup_mikado_get_global_variables' );
}

if ( ! function_exists( 'boostup_mikado_per_page_js_variables' ) ) {
	/**
	 * Outputs global JS variable that holds page settings
	 */
	function boostup_mikado_per_page_js_variables() {
		$per_page_js_vars = apply_filters( 'boostup_mikado_filter_per_page_js_vars', array() );

		wp_localize_script( 'boostup-mikado-modules', 'mkdfPerPageVars', array(
			'vars' => $per_page_js_vars
		) );
	}

	add_action( 'wp_enqueue_scripts', 'boostup_mikado_per_page_js_variables' );
}

if ( ! function_exists( 'boostup_mikado_content_elem_style_attr' ) ) {
	/**
	 * Defines filter for adding custom styles to content HTML element
	 */
	function boostup_mikado_content_elem_style_attr() {
		$styles = apply_filters( 'boostup_mikado_filter_content_elem_style_attr', array() );

		boostup_mikado_inline_style( $styles );
	}
}

if ( ! function_exists( 'boostup_mikado_core_plugin_installed' ) ) {
	/**
	 * Function that checks if Mikado Core plugin installed
	 * @return bool
	 */
	function boostup_mikado_core_plugin_installed() {
		return defined( 'BOOSTUP_CORE_VERSION' );
	}
}

if ( ! function_exists( 'boostup_mikado_is_woocommerce_installed' ) ) {
	/**
	 * Function that checks if Woocommerce plugin installed
	 * @return bool
	 */
	function boostup_mikado_is_woocommerce_installed() {
		return function_exists( 'is_woocommerce' );
	}
}

if ( ! function_exists( 'boostup_mikado_visual_composer_installed' ) ) {
	/**
	 * Function that checks if Visual Composer plugin installed
	 * @return bool
	 */
	function boostup_mikado_visual_composer_installed() {
		return class_exists( 'WPBakeryVisualComposerAbstract' );
	}
}

if ( ! function_exists( 'boostup_mikado_revolution_slider_installed' ) ) {
	/**
	 * Function that checks if Revolution Slider plugin installed
	 * @return bool
	 */
	function boostup_mikado_revolution_slider_installed() {
		return class_exists( 'RevSliderFront' );
	}
}

if ( ! function_exists( 'boostup_mikado_contact_form_7_installed' ) ) {
	/**
	 * Function that checks if Contact Form 7 plugin installed
	 * @return bool
	 */
	function boostup_mikado_contact_form_7_installed() {
		return defined( 'WPCF7_VERSION' );
	}
}

if ( ! function_exists( 'boostup_mikado_is_wpml_installed' ) ) {
	/**
	 * Function that checks if WPML plugin installed
	 * @return bool
	 */
	function boostup_mikado_is_wpml_installed() {
		return defined( 'ICL_SITEPRESS_VERSION' );
	}
}

if ( ! function_exists( 'boostup_mikado_is_gutenberg_installed' ) ) {
	/**
	 * Function that checks if Gutenberg plugin installed
	 * @return bool
	 */
	function boostup_mikado_is_gutenberg_installed() {
		return function_exists( 'is_gutenberg_page' ) && is_gutenberg_page();
	}
}

if ( ! function_exists( 'boostup_mikado_max_image_width_srcset' ) ) {
	/**
	 * Set max width for srcset to 1920
	 *
	 * @return int
	 */
	function boostup_mikado_max_image_width_srcset() {
		return 1920;
	}
	
	add_filter( 'max_srcset_image_width', 'boostup_mikado_max_image_width_srcset' );
}

if ( ! function_exists( 'boostup_mikado_get_module_part' ) ) {
    function boostup_mikado_get_module_part( $module ) {
        return $module;
    }
}


if ( ! function_exists( 'boostup_mikado_has_dashboard_shortcodes' ) ) {
	/**
	 * Function that checks if current page has at least one of dashboard shortcodes added
	 * @return bool
	 */
	function boostup_mikado_has_dashboard_shortcodes() {
		$dashboard_shortcodes = array();

		$dashboard_shortcodes = apply_filters( 'boostup_mikado_filter_dashboard_shortcodes_list', $dashboard_shortcodes );

		foreach ( $dashboard_shortcodes as $dashboard_shortcode ) {
			$has_shortcode = boostup_mikado_has_shortcode( $dashboard_shortcode );

			if ( $has_shortcode ) {
				return true;
			}
		}

		return false;
	}
}